<?php
/**
 * Redeem Coupon
 *
 * @package     AutomatorWP\Integrations\GamiPress\Triggers\Redeem_Coupon
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_GamiPress_Redeem_Coupon extends AutomatorWP_Integration_Trigger {

    public $integration = 'gamipress';
    public $trigger = 'gamipress_redeem_coupon';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User redeems a coupon where code matches with a specific condition', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>redeems a coupon</strong> where code matches with a specific condition', 'automatorwp-pro' ),
            /* translators: %1$s: Points amount. %2$s: Post title. */
            'edit_label'        => sprintf( __( 'User redeems a coupon where code %1$s %2$s %3$s', 'automatorwp-pro' ), '{condition}', '{code}', '{times}' ),
            /* translators: %1$s: Points amount. %2$s: Post title. */
            'log_label'         => sprintf( __( 'User redeems a coupon where code %1$s %2$s %3$s', 'automatorwp-pro' ), '{condition}', '{code}', '{times}' ),
            'action'            => 'gamipress_coupons_user_redeem_coupon',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 3,
            'options'           => array(
                'condition' => automatorwp_utilities_string_condition_option(),
                'code' => array(
                    'from' => 'code',
                    'default' => __( 'code', 'automatorwp-pro' ),
                    'fields' => array(
                        'code' => array(
                            'name' => __( 'Coupon Code:', 'automatorwp-pro' ),
                            'desc' => __( 'Required coupon code.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'default' => ''
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                array(
                    'coupon_code' => array(
                        'label'     => __( 'Coupon Code', 'automatorwp-pro' ),
                        'type'      => 'text',
                        'preview'   => __( 'G4M1PR3SS', 'automatorwp-pro' ),
                    ),
                ),
                automatorwp_utilities_post_tags()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int       $user_id        User that redeemed the coupon
     * @param stdClass  $coupon         Coupon stdClass object
     * @param array     $coupon_rewards Coupon rewards array
     */
    public function listener( $user_id, $coupon, $coupon_rewards ) {

        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'coupon_code'   => $coupon->code,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if coupon code is not received
        if( ! isset( $event['coupon_code'] ) ) {
            return false;
        }

        // Don't deserve if coupon code doesn't match with the trigger option
        if( ! automatorwp_condition_matches( $event['coupon_code'], $trigger_options['code'], $trigger_options['condition'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */
    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        // Trigger tags replacements
        add_filter( 'automatorwp_trigger_tags_replacements', array( $this, 'tags_replacements' ), 10, 4 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */
    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['coupon_code'] = ( isset( $event['coupon_code'] ) ? $event['coupon_code'] : '' );

        return $log_meta;

    }

    /**
     * Filter to setup custom trigger tags replacements
     *
     * Note: Post and times tags replacements are already passed
     *
     * @since 1.0.0
     *
     * @param array     $replacements   The trigger replacements
     * @param stdClass  $trigger        The trigger object
     * @param int       $user_id        The user ID
     * @param stdClass  $log            The last trigger log object
     *
     * @return array
     */
    public function tags_replacements( $replacements, $trigger, $user_id, $log ) {

        // Bail if trigger type don't match this trigger
        if( $trigger->type !== $this->trigger ) {
            return $replacements;
        }

        // Times replacement by default
        $replacements['coupon_code'] = ct_get_object_meta( $log->id, 'coupon_code', true );

        return $replacements;

    }

}

new AutomatorWP_GamiPress_Redeem_Coupon();